import { system, world } from '@minecraft/server'
import { Adult, Baby, Item, SpawnEgg } from '../item/Hamster'
import { Helper, Vector3 } from "../utils/Helper"
const BLOCK_TAG = 'sf_hba:hamster_hut'
const BLOCK_ENTITY = 'sf_hba:hamster_hut'
const SEARCH_BED_COOLDOWN = 'sf_nba:hamster_add_search_hamster_hut_delay'
const CAN_NOT_MOVE = 'sf_nba:cannot_move'
const WAKE = 'sf_nba:hamster_is_awake'
const MOVE_AWAY_FROM_INTERACTIVE_BLOCK = 'sf_nba:hamster_move_away_from_interactive_block'
function wakeUpHamster(entity) {
	const hamster = getHamsterSleeping(entity)
	if (!hamster) return undefined
	const rideable = entity.getComponent('rideable')
	hamster.triggerEvent(WAKE)
	rideable.ejectRider(hamster)
	return hamster
}
function getHamsterSleeping(entity) {
	const rideable = entity.getComponent('rideable')
	if (!rideable) return undefined
	const rider = rideable.getRiders()[0]
	if (!rider) return undefined
	if (rider.typeId.startsWith('sf_hba')) return undefined
	return rider
}
function getBlockEntityAt(block, dimension) {
	const entity = dimension.getEntities({
		type: BLOCK_ENTITY,
		closest: 1,
		location: block.bottomCenter()
	})
	return entity
}
function hasHamsterSleeping(entity) {
	const rider = entity.getComponent('rideable')
	if (!rider) return true
	return rider.getRiders().length == 1
}
function bottomCenterVector3(pos) {
	return new Vector3(
		Math.floor(pos.x) + 0.5,
		Math.floor(pos.y),
		Math.floor(pos.z) + 0.5
	)
}
class HamsterHut {
	constructor() {
		this.init()
	}
	init() {
		world.beforeEvents.playerInteractWithBlock.subscribe(this.interact.bind(this))
		world.beforeEvents.playerBreakBlock.subscribe(this.onPlayerDestroy.bind(this))
	}
	interact(e) {
		const { block, itemStack, player } = e
		if (!block.hasTag(BLOCK_TAG)) return
		const blockEntity = getBlockEntityAt(block, block.dimension)[0]
		if ([...Adult, ...Baby, ...Item, ...SpawnEgg].includes(itemStack?.typeId)) {
			system.run(() => {
				const pos = bottomCenterVector3(Vector3.fromMinecraftVector3(blockEntity.location))
				const hamster = blockEntity.dimension.spawnEntity("sf_nba:hamster", pos.add(new Vector3(0, 0.5, 0)))
				const tameComponent = hamster.getComponent('tameable')
				hamster.teleport(pos)
				hamster.triggerEvent(CAN_NOT_MOVE)
				Helper.decreaseMainhandItemStack(player, 1, itemStack)
				if (SpawnEgg.includes(itemStack.typeId)) {
					const variant = SpawnEgg.indexOf(itemStack.typeId)
					try {
						tameComponent?.tame(player)
						hamster.setProperty('sf_nba:variant', variant)
					} catch { }
				}
				if ([...Adult, ...Baby, ...Item].includes(itemStack.typeId)) {
					const hamsterArray = itemStack.typeId.includes('baby') ? Baby : itemStack.typeId.includes('adult') ? Adult : Item
					const ownerId = itemStack.getDynamicProperty('sf_nba:owner_id')
					const health = itemStack.getDynamicProperty('sf_nba:health')
					const variant = hamsterArray.indexOf(itemStack.typeId)
					const dye = itemStack.getDynamicProperty('sf_nba:dye') ?? 15
					const newOwner = ownerId ? world.getPlayers().filter(p => p.id == ownerId)[0] : player
					const newHealth = health ? health : hamster.getComponent('health')?.defaultValue
					const age = [...Item, ...Adult].includes(itemStack.typeId) ? 'sf_nba:adult' : 'sf_nba:baby'
					try {
						tameComponent?.tame(newOwner)
						hamster.setDynamicProperty('sf_nba:owner_id', player.id)
						hamster.triggerEvent(age)
						hamster.getComponent('health')?.setCurrentValue(newHealth)
						hamster.setProperty('sf_nba:variant', variant)
						hamster.setProperty('sf_nba:dye', dye)
					} catch { }
				}
			})
			e.cancel = true
			return
		}
		system.run(() => {
			const hamster = wakeUpHamster(blockEntity)
			if (!hamster) return
			hamster.triggerEvent(MOVE_AWAY_FROM_INTERACTIVE_BLOCK)
		})
	}
	onPlayerDestroy(e) {
		const { block, dimension, itemStack, player } = e
		if (!block.hasTag(BLOCK_TAG)) return
		system.run(() => {
			const blockEntity = getBlockEntityAt(block, dimension)[0]
			if (!blockEntity) return
			if (!hasHamsterSleeping(blockEntity)) return
			const hamster = wakeUpHamster(blockEntity)
			if (hamster?.typeId.startsWith('sf_nba')) {
				hamster.triggerEvent(SEARCH_BED_COOLDOWN)
			}
		})
	}
}
new HamsterHut()